//*************************************************************************************************
//
//	Description:
//		Basic billboarding shader
//		
//		
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Matt Hobbs
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		Matt             09/08/2006  Created
//		TMann			 08/02/2007	 1.2			Added GL/PS3
//
//	<TABLE>
//
//*************************************************************************************************
#define _SSAO_READY_

//-----------------------------------------------------------------------
//
// Lighting
// 

#if defined(_XBOX) && !defined(_TOOLS_COMPILATION_)
#include "D:\Render\Shaders\stddefs.fxh"
#include "D:\Render\Shaders\lighting_globals.fxh"
#else
#include "..\..\..\Render\Shaders\stddefs.fxh"
#include "..\..\..\Render\Shaders\lighting_globals.fxh"
#endif
DECLARE_LIGHTING_PARAMS

#if defined(_PS3_)
#define _MINFILTER	LinearMipMapLinear
#else
#define _MINFILTER	Linear
#define SET_NO_ANISOTROPY MaxAnisotropy = 1;
#endif

//-----------------------------------------------------------------------
//
// defines used by z-particles, for testing
//

//#define VISUALISE_ZBUFFER
//#define VISUALISE_DEPTH_DIFFERENCE
//#define VISUALISE_PARTICLE_DEPTH

//-----------------------------------------------------------------------
//
// defines used by z-particles, PARTICLE_PIXEL_DEPTH_IN_CLIP_PLANE_RANGE must be defined for normal operation
//
#define PARTICLE_PIXEL_DEPTH_IN_CLIP_PLANE_RANGE
//#define PARTICLE_PIXEL_DEPTH_IN_ZBUFFER_RANGE

//#define CONSTANT_DEPTH_FADE
//#define PIXEL_OFFSET_DEPTH_FADE
#define PIXEL_ALPHA_DEPTH_FADE

//-----------------------------------------------------------------------
//
// Transforms
//
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;

float4x4 projMatrix : Projection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;

float3 worldCameraPos : WorldCameraPosition
<
	bool appEdit = false;
>;




//-----------------------------------------------------------------------
//
// Misc params
//
/*float4 colourModulation
<
	bool appEdit = true;
> = float4(1.0f, 1.0f, 1.0f, 1.0f);*/

float startCamFadeDistToVertex
<
	bool appEdit = true;
> = 0.0f;

float endCamFadeDistToVertex
<
	bool appEdit = true;
> = 0.0f;

// near, far, far * near, far - near
float4 nearFarClip
<
	bool appEdit = true;
>;

float vertexLightingScaler
<
	bool appEdit = true;
> = 0.0f;

//-----------------------------------------------------------------------
//
// Textures
//
texture diffuseTexture0 : TEXTURE							// Diffuse colour in RGB, translucency in alpha
<
	string UIName = "Diffuse Texture 1";
	bool appEdit = true;
>;

texture diffuseTexture1 : TEXTURE							// Diffuse colour in RGB, translucency in alpha
<
	string UIName = "Diffuse Texture 2";
	bool appEdit = true;
>;

texture depthTexture : TEXTURE							
<
	string UIName = "Depth Texture";
	bool appEdit = true;
>;



//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D diffuseMap0 : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture = "diffuseTexture0"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
>
= sampler_state
{
	Texture = < diffuseTexture0 >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Clamp;
	AddressV  = Clamp;
#endif
};

sampler2D diffuseMap1 : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture = "diffuseTexture1"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
>
= sampler_state
{
	Texture = < diffuseTexture1 >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Clamp;
	AddressV  = Clamp;
#endif
};

sampler2D depthMap : SAMPLER
<
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture = "depthTexture"; 
	string MinFilter = "Point";
	string MagFilter = "Point";
	string MipFilter = "None";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	int MipMapLODBias = 0;
>
= sampler_state
{
	Texture = < depthTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = Point;
	MagFilter = Point;
	MipFilter = None;
	AddressU  = Clamp;
	AddressV  = Clamp;
#if defined(_PS3_)
	LODBias = 0;
#else		// ! _PS3_
	MipMapLODBias = 0;
#endif	// ! _PS3_
	SET_NO_ANISOTROPY
#endif	// SET_FX_SAMPLER_STATES
};






//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

float4 c_texAnimMatrixScaleTrans
<
	bool appEdit = true;
>;

float2 c_numAnimFrames
<
	bool appEdit = true;
>;

float c_totalAnimFrames
<
	bool appEdit = true;
>;

float2 c_recipNumAnimFrames
<
	bool appEdit = true;
>;

float4x4 c_uvArray
<
	bool appEdit = true; 
>;

// TODO: put scale into vertex shader const when supported render-side
struct VSINPUT
{
	float4 positionVertIdx				: POSITION;				// 3D position and vertex index for quad particle in [0..3] 								
	float3 widthHeightRot				: NORMAL;				// width, height, rotation 
	float animT							: TEXCOORD0;			// value in [0..1] which is the normalised time into the texture animation
	float4 colour						: COLOR0;				// vertex colour
};

// TODO: pack texlerp if poss
struct VSOUTPUT
{
	float4 position						: POSITION;												
	float4 texCoord01					: TEXCOORD0;
	float texLerp						: TEXCOORD1;
	float4 colour						: TEXCOORD2;												
};

// TODO: pack texlerp if poss
struct VSOUTPUT_DEPTH
{
	float4 position						: POSITION;												
	float4 texCoord01					: TEXCOORD0;
	float texLerp						: TEXCOORD1;
	float3 depthTexCoordPixelDepth		: TEXCOORD2;											
	float4 colour						: TEXCOORD3;	
#ifdef PIXEL_OFFSET_DEPTH_FADE
	float2 vertOfs						: TEXCOORD4;	
#endif //#ifdef PIXEL_OFFSET_DEPTH_FADE
#ifdef PIXEL_ALPHA_DEPTH_FADE
	float particleRadius				: TEXCOORD4;	
#endif //#ifdef PIXEL_ALPHA_DEPTH_FADE
};



//-----------------------------------------------------------------------
//
// Vertex shader code
//

VSOUTPUT VShader( VSINPUT _input )
{
	VSOUTPUT _output;
	
	float sn, cs;
	float3x2 texAnimMatrix;

	// get rotation params for 2D matrix mul (texcoord rotation of quad, rather than vertex rotation)
	sn = sin(_input.widthHeightRot.z);
	cs = cos(_input.widthHeightRot.z);
	
	// get homogeneous uv for this vert from the constant params
	int vertexIdx = (int)_input.positionVertIdx.w;
	float3 texCoord = float3(c_uvArray[vertexIdx].x, c_uvArray[vertexIdx].y, 1.0f);
	
	// set up rotation and scale portions of uv anim matrix
	texAnimMatrix._m00 = cs * c_texAnimMatrixScaleTrans.x;
	texAnimMatrix._m01 = -sn * c_texAnimMatrixScaleTrans.y;
	texAnimMatrix._m10 = sn * c_texAnimMatrixScaleTrans.x;
	texAnimMatrix._m11 = cs * c_texAnimMatrixScaleTrans.y;
		
	// get corner offset in clip space	
	float4 cornerOfs = mul( float4(_input.widthHeightRot.x, _input.widthHeightRot.y, 0.0f, 0.0f), projMatrix );

	// get centre position in clip space and combine with offset
	float4 inPos = float4( _input.positionVertIdx.xyz, 1.0f );
	_output.position = mul( inPos, worldviewproj );
	_output.position += cornerOfs;
	
	// perform texture animation calculation - find which frames in u+v directions we are in for this anim time
	float uFrame0, vFrame0;
	float uFrame1, vFrame1;
	float animFrameT;
	animFrameT = _input.animT * c_totalAnimFrames; 

	// texture 0
	uFrame0 = fmod(animFrameT, c_numAnimFrames.x); 
	vFrame0 = animFrameT - uFrame0;
	uFrame0 = floor(uFrame0);
	vFrame0 *= c_recipNumAnimFrames.y;

	// texture 1
	animFrameT += Rv(1.0);
	if(animFrameT > c_totalAnimFrames)
	{
		// wrap around so texcoords don't go out of bounds of texture
		animFrameT -= c_totalAnimFrames;
	}
	uFrame1 = fmod(animFrameT, c_numAnimFrames.x); 
	vFrame1 = animFrameT - uFrame1;
	uFrame1 = floor(uFrame1);
	vFrame1 *= c_recipNumAnimFrames.y;
	
	// get back to normalised values 
	uFrame0 *= c_recipNumAnimFrames.x;
	vFrame0 *= c_recipNumAnimFrames.y;
	uFrame1 *= c_recipNumAnimFrames.x;
	vFrame1 *= c_recipNumAnimFrames.y;

	// apply translation to map texcoords back to [0..1] from [-1..1] (post rotation)
	uFrame0 += c_texAnimMatrixScaleTrans.z;
	vFrame0 += c_texAnimMatrixScaleTrans.w;
	uFrame1 += c_texAnimMatrixScaleTrans.z;
	vFrame1 += c_texAnimMatrixScaleTrans.w;
	
	// calc texcoord for texture 0 (use corresponding translation offset)
	texAnimMatrix._m20 = uFrame0;
	texAnimMatrix._m21 = vFrame0;
	_output.texCoord01.xy = mul( texCoord, texAnimMatrix);
		
	// calc texcoord for texture 1 (use corresponding translation offset)
	texAnimMatrix._m20 = uFrame1;
	texAnimMatrix._m21 = vFrame1;
	_output.texCoord01.zw = mul( texCoord, texAnimMatrix);
	
	// apply lighting
	_output.colour = float4(1.0f, 1.0f, 1.0f, 1.0f);
	// non unit normal limits the amount of directional lighting (we're not a solid surface) 
	float3 normal = float3(0.0f, vertexLightingScaler, 0.0f);	
	float4 worldPos = mul(inPos, world);
	DO_VERTEX_LIGHTING( worldPos, normal, _output.colour )
	_output.colour *= _input.colour;
	
	// TODO: reinstate if needed (old col mod vshader / technique removed)
	// modulate colour by global factor
	//_output.colour *= colourModulation;
		
	// calc transition from one frame to the next
	_output.texLerp = frac(_input.animT * c_totalAnimFrames);
		
	return _output;
}

VSOUTPUT VShaderCamFade( VSINPUT _input )
{
	VSOUTPUT _output;
	
	float sn, cs;
	float3x2 texAnimMatrix;

	// get rotation params for 2D matrix mul (texcoord rotation of quad, rather than vertex rotation)
	sn = sin(_input.widthHeightRot.z);
	cs = cos(_input.widthHeightRot.z);
	
	// TODO: make into BONEINDEX
	// get homogeneous uv for this vert from the constant params
	int vertexIdx = (int)_input.positionVertIdx.w;
	float3 texCoord = float3(c_uvArray[vertexIdx].x, c_uvArray[vertexIdx].y, 1.0f);
	
	// set up rotation and scale portions of uv anim matrix
	texAnimMatrix._m00 = cs * c_texAnimMatrixScaleTrans.x;
	texAnimMatrix._m01 = -sn * c_texAnimMatrixScaleTrans.y;
	texAnimMatrix._m10 = sn * c_texAnimMatrixScaleTrans.x;
	texAnimMatrix._m11 = cs * c_texAnimMatrixScaleTrans.y;
		
	// get corner offset in clip space	
	float4 cornerOfs = mul( float4(_input.widthHeightRot.x, _input.widthHeightRot.y, 0.0f, 0.0f), projMatrix );

	// get centre position in clip space and combine with offset
	float4 inPos = float4( _input.positionVertIdx.xyz, 1.0f );
	_output.position = mul( inPos, worldviewproj );
	_output.position += cornerOfs;
	
	// perform texture animation calculation - find which frames in u+v directions we are in for this anim time
	float uFrame0, vFrame0;
	float uFrame1, vFrame1;
	float animFrameT;
	animFrameT = _input.animT * c_totalAnimFrames; 

	// texture 0
	uFrame0 = fmod(animFrameT, c_numAnimFrames.x); 
	vFrame0 = animFrameT - uFrame0;
	uFrame0 = floor(uFrame0);
	vFrame0 *= c_recipNumAnimFrames.y;

	// texture 1
	animFrameT += Rv(1.0);
	if(animFrameT > c_totalAnimFrames)
	{
		// wrap around so texcoords don't go out of bounds of texture
		animFrameT -= c_totalAnimFrames;
	}
	uFrame1 = fmod(animFrameT, c_numAnimFrames.x); 
	vFrame1 = animFrameT - uFrame1;
	uFrame1 = floor(uFrame1);
	vFrame1 *= c_recipNumAnimFrames.y;
	
	// get back to normalised values 
	uFrame0 *= c_recipNumAnimFrames.x;
	vFrame0 *= c_recipNumAnimFrames.y;
	uFrame1 *= c_recipNumAnimFrames.x;
	vFrame1 *= c_recipNumAnimFrames.y;

	// apply translation to map texcoords back to [0..1] from [-1..1] (post rotation)
	uFrame0 += c_texAnimMatrixScaleTrans.z;
	vFrame0 += c_texAnimMatrixScaleTrans.w;
	uFrame1 += c_texAnimMatrixScaleTrans.z;
	vFrame1 += c_texAnimMatrixScaleTrans.w;
	
	// calc texcoord for texture 0 (use corresponding translation offset)
	texAnimMatrix._m20 = uFrame0;
	texAnimMatrix._m21 = vFrame0;
	_output.texCoord01.xy = mul( texCoord, texAnimMatrix);
		
	// calc texcoord for texture 1 (use corresponding translation offset)
	texAnimMatrix._m20 = uFrame1;
	texAnimMatrix._m21 = vFrame1;
	_output.texCoord01.zw = mul( texCoord, texAnimMatrix);
	
	// apply lighting
	_output.colour = float4(1.0f, 1.0f, 1.0f, 1.0f);
	// non unit normal limits the amount of directional lighting (we're not a solid surface) 
	float3 normal = float3(0.0f, vertexLightingScaler, 0.0f);	
	float4 worldPos = mul(inPos, world);
	DO_VERTEX_LIGHTING( worldPos, normal, _output.colour )
	_output.colour *= _input.colour;
	
	// TODO: reinstate if needed (old col mod vshader / technique removed)
	// modulate colour by global factor
	//_output.colour *= colourModulation;
	
	// perform cam fade calc
	float3 diff = float3(worldPos.xyz - worldCameraPos);
	float dist = length(diff);
	float alphaFactor = 1.0f - saturate((startCamFadeDistToVertex - (dist - endCamFadeDistToVertex)) / startCamFadeDistToVertex);
	_output.colour.a *= alphaFactor;
		
	// calc transition from one frame to the next
	_output.texLerp = frac(_input.animT * c_totalAnimFrames);
		
	return _output;
}

VSOUTPUT_DEPTH VShaderZ( VSINPUT _input )
{
	VSOUTPUT_DEPTH _output;
	
	float sn, cs;
	float3x2 texAnimMatrix;

	// get rotation params for 2D matrix mul (texcoord rotation of quad, rather than vertex rotation)
	sn = sin(_input.widthHeightRot.z);
	cs = cos(_input.widthHeightRot.z);
	
	// TODO: make into BONEINDEX
	// get homogeneous uv for this vert from the constant params
	int vertexIdx = (int)_input.positionVertIdx.w;
	float3 texCoord = float3(c_uvArray[vertexIdx].x, c_uvArray[vertexIdx].y, 1.0f);
	
	// set up rotation and scale portions of uv anim matrix
	texAnimMatrix._m00 = cs * c_texAnimMatrixScaleTrans.x;
	texAnimMatrix._m01 = -sn * c_texAnimMatrixScaleTrans.y;
	texAnimMatrix._m10 = sn * c_texAnimMatrixScaleTrans.x;
	texAnimMatrix._m11 = cs * c_texAnimMatrixScaleTrans.y;
		
	// get corner offset in clip space	
	float4 cornerOfs = mul( float4(_input.widthHeightRot.x, _input.widthHeightRot.y, 0.0f, 0.0f), projMatrix );

	// get centre position in clip space and combine with offset
	float4 inPos = float4( _input.positionVertIdx.xyz, 1.0f );
	_output.position = mul( inPos, worldviewproj );
	_output.position += cornerOfs;
	
#ifdef PIXEL_OFFSET_DEPTH_FADE	
	_output.vertOfs.x = _input.widthHeightRot.x;
	_output.vertOfs.y = _input.widthHeightRot.y;
#endif //#ifdef PIXEL_OFFSET_DEPTH_FADE
#ifdef PIXEL_ALPHA_DEPTH_FADE
	_output.particleRadius = sqrt(_input.widthHeightRot.x * _input.widthHeightRot.x + _input.widthHeightRot.y * _input.widthHeightRot.y);
#endif //#ifdef PIXEL_ALPHA_DEPTH_FADE
	
	// perform texture animation calculation - find which frames in u+v directions we are in for this anim time
	float uFrame0, vFrame0;
	float uFrame1, vFrame1;
	float animFrameT;
	animFrameT = _input.animT * c_totalAnimFrames; 

	// texture 0
	uFrame0 = fmod(animFrameT, c_numAnimFrames.x); 
	vFrame0 = animFrameT - uFrame0;
	uFrame0 = floor(uFrame0);
	vFrame0 *= c_recipNumAnimFrames.y;

	// texture 1
	animFrameT += Rv(1.0);
	if(animFrameT > c_totalAnimFrames)
	{
		// wrap around so texcoords don't go out of bounds of texture
		animFrameT -= c_totalAnimFrames;
	}
	uFrame1 = fmod(animFrameT, c_numAnimFrames.x); 
	vFrame1 = animFrameT - uFrame1;
	uFrame1 = floor(uFrame1);
	vFrame1 *= c_recipNumAnimFrames.y;
	
	// get back to normalised values 
	uFrame0 *= c_recipNumAnimFrames.x;
	vFrame0 *= c_recipNumAnimFrames.y;
	uFrame1 *= c_recipNumAnimFrames.x;
	vFrame1 *= c_recipNumAnimFrames.y;

	// apply translation to map texcoords back to [0..1] from [-1..1] (post rotation)
	uFrame0 += c_texAnimMatrixScaleTrans.z;
	vFrame0 += c_texAnimMatrixScaleTrans.w;
	uFrame1 += c_texAnimMatrixScaleTrans.z;
	vFrame1 += c_texAnimMatrixScaleTrans.w;
	
	// calc texcoord for texture 0 (use corresponding translation offset)
	texAnimMatrix._m20 = uFrame0;
	texAnimMatrix._m21 = vFrame0;
	_output.texCoord01.xy = mul( texCoord, texAnimMatrix);
		
	// calc texcoord for texture 1 (use corresponding translation offset)
	texAnimMatrix._m20 = uFrame1;
	texAnimMatrix._m21 = vFrame1;
	_output.texCoord01.zw = mul( texCoord, texAnimMatrix);
	
	// apply lighting
	_output.colour = float4(1.0f, 1.0f, 1.0f, 1.0f);
	// non unit normal limits the amount of directional lighting (we're not a solid surface) 
	float3 normal = float3(0.0f, vertexLightingScaler, 0.0f);	
	float4 worldPos = mul(inPos, world);
	DO_VERTEX_LIGHTING( worldPos, normal, _output.colour )
	_output.colour *= _input.colour;
	
	// TODO: reinstate if needed (old col mod vshader / technique removed)
	// modulate colour by global factor
	//_output.colour *= colourModulation;
		
	// calc transition from one frame to the next
	_output.texLerp = frac(_input.animT * c_totalAnimFrames);
	
	// get uv's projected into 0..1 range, for looking up into zbuffer
	// NOTE: tried using tex2dproj instead of /w, but doesnt work. I think cos verts are not in world space, but constructed in screen space?
	float recipW = 1.0f / _output.position.w;
	_output.depthTexCoordPixelDepth.x = (_output.position.x * recipW * 0.5f) + 0.5f;
	_output.depthTexCoordPixelDepth.y = (-_output.position.y * recipW * 0.5f) + 0.5f;
	
#ifdef PARTICLE_PIXEL_DEPTH_IN_CLIP_PLANE_RANGE
	// transformed w is camera space z
	_output.depthTexCoordPixelDepth.z = _output.position.w;
#endif //#ifdef PARTICLE_PIXEL_DEPTH_IN_CLIP_PLANE_RANGE
	
#ifdef PARTICLE_PIXEL_DEPTH_IN_ZBUFFER_RANGE
	// For comparing with standard depth buffer
	_output.depthTexCoordPixelDepth.z = _output.position.z * recipW;
#endif //#ifdef PARTICLE_PIXEL_DEPTH_IN_ZBUFFER_RANGE
		
	return _output;
}

VSOUTPUT_DEPTH VShaderCamFadeZ( VSINPUT _input )
{
	VSOUTPUT_DEPTH _output;
	
	float sn, cs;
	float3x2 texAnimMatrix;
	float2x2 vertRotMatrix;

	// get rotation params for 2D matrix mul (texcoord rotation of quad, rather than vertex rotation)
	sn = sin(_input.widthHeightRot.z);
	cs = cos(_input.widthHeightRot.z);
	
	// TODO: make into BONEINDEX
	// get homogeneous uv for this vert from the constant params
	int vertexIdx = (int)_input.positionVertIdx.w;
	float3 texCoord = float3(c_uvArray[vertexIdx].x, c_uvArray[vertexIdx].y, 1.0f);
	
	// set up rotation and scale portions of uv anim matrix
	texAnimMatrix._m00 = c_texAnimMatrixScaleTrans.x;
	texAnimMatrix._m01 = 0.0f;
	texAnimMatrix._m10 = 0.0f;
	texAnimMatrix._m11 = c_texAnimMatrixScaleTrans.y;
	
	vertRotMatrix._m00 = cs;
	vertRotMatrix._m01 = -sn;
	vertRotMatrix._m10 = sn;
	vertRotMatrix._m11 = cs;
		
	// get corner offset in clip space	
	float2 ofs = float2(_input.widthHeightRot.x, _input.widthHeightRot.y);
	ofs = mul( ofs, vertRotMatrix);
	float4 cornerOfs = mul( float4(ofs.x, ofs.y, 0.0f, 0.0f), projMatrix );
	
	// get centre position in clip space and combine with offset
	float4 inPos = float4( _input.positionVertIdx.xyz, 1.0f );
	_output.position = mul( inPos, worldviewproj );
	_output.position += cornerOfs;

#ifdef PIXEL_OFFSET_DEPTH_FADE	
	_output.vertOfs.x = _input.widthHeightRot.x;
	_output.vertOfs.y = _input.widthHeightRot.y;
#endif //#ifdef PIXEL_OFFSET_DEPTH_FADE
#ifdef PIXEL_ALPHA_DEPTH_FADE
	_output.particleRadius = sqrt(_input.widthHeightRot.x * _input.widthHeightRot.x + _input.widthHeightRot.y * _input.widthHeightRot.y);
#endif //#ifdef PIXEL_ALPHA_DEPTH_FADE
	
	// perform texture animation calculation - find which frames in u+v directions we are in for this anim time
	float uFrame0, vFrame0;
	float uFrame1, vFrame1;
	float animFrameT;
	animFrameT = _input.animT * c_totalAnimFrames; 

	// texture 0
	uFrame0 = fmod(animFrameT, c_numAnimFrames.x); 
	vFrame0 = animFrameT - uFrame0;
	uFrame0 = floor(uFrame0);
	vFrame0 *= c_recipNumAnimFrames.y;

	// texture 1
	animFrameT += Rv(1.0);
	if(animFrameT > c_totalAnimFrames)
	{
		// wrap around so texcoords don't go out of bounds of texture
		animFrameT -= c_totalAnimFrames;
	}
	uFrame1 = fmod(animFrameT, c_numAnimFrames.x); 
	vFrame1 = animFrameT - uFrame1;
	uFrame1 = floor(uFrame1);
	vFrame1 *= c_recipNumAnimFrames.y;
	
	// get back to normalised values 
	uFrame0 *= c_recipNumAnimFrames.x;
	vFrame0 *= c_recipNumAnimFrames.y;
	uFrame1 *= c_recipNumAnimFrames.x;
	vFrame1 *= c_recipNumAnimFrames.y;

	// apply translation to map texcoords back to [0..1] from [-1..1] (post rotation)
	uFrame0 += c_texAnimMatrixScaleTrans.z;
	vFrame0 += c_texAnimMatrixScaleTrans.w;
	uFrame1 += c_texAnimMatrixScaleTrans.z;
	vFrame1 += c_texAnimMatrixScaleTrans.w;
	
	// calc texcoord for texture 0 (use corresponding translation offset)
	texAnimMatrix._m20 = uFrame0;
	texAnimMatrix._m21 = vFrame0;
	_output.texCoord01.xy = mul( texCoord, texAnimMatrix);
		
	// calc texcoord for texture 1 (use corresponding translation offset)
	texAnimMatrix._m20 = uFrame1;
	texAnimMatrix._m21 = vFrame1;
	_output.texCoord01.zw = mul( texCoord, texAnimMatrix);
	
	// apply lighting
	_output.colour = float4(1.0f, 1.0f, 1.0f, 1.0f);
	// non unit normal limits the amount of directional lighting (we're not a solid surface) 
	float3 normal = float3(0.0f, vertexLightingScaler, 0.0f);	
	float4 worldPos = mul(inPos, world);
	DO_VERTEX_LIGHTING( worldPos, normal, _output.colour )
	_output.colour *= _input.colour;
	
	// TODO: reinstate if needed (old col mod vshader / technique removed)
	// modulate colour by global factor
	//_output.colour *= colourModulation;
	
	// perform cam fade calc
	float3 diff = float3(worldPos.xyz - worldCameraPos);
	float dist = length(diff);
	float alphaFactor = 1.0f - saturate((startCamFadeDistToVertex - (dist - endCamFadeDistToVertex)) / startCamFadeDistToVertex);
	_output.colour.a *= alphaFactor;
		
	// calc transition from one frame to the next
	_output.texLerp = frac(_input.animT * c_totalAnimFrames);
	
	// get uv's projected into 0..1 range, for looking up into zbuffer
	// NOTE: tried using tex2dproj instead of /w, but doesnt work. I think cos verts are not in world space, but constructed in screen space?
	float recipW = 1.0f / _output.position.w;
	_output.depthTexCoordPixelDepth.x = (_output.position.x * recipW * 0.5f) + 0.5f;
	_output.depthTexCoordPixelDepth.y = (-_output.position.y * recipW * 0.5f) + 0.5f;
	
#ifdef PARTICLE_PIXEL_DEPTH_IN_CLIP_PLANE_RANGE
	// transformed w is camera space z
	_output.depthTexCoordPixelDepth.z = _output.position.w;
#endif //#ifdef PARTICLE_PIXEL_DEPTH_IN_CLIP_PLANE_RANGE
	
#ifdef PARTICLE_PIXEL_DEPTH_IN_ZBUFFER_RANGE
	// For comparing with standard depth buffer
	_output.depthTexCoordPixelDepth.z = _output.position.z * recipW;
#endif //#ifdef PARTICLE_PIXEL_DEPTH_IN_ZBUFFER_RANGE
		
	return _output;
}







/*
// OLD / ALTERNATIVE ANIM CODE - 2ND PART WORTH TRYING
/*uFrame0 = nsMaths::Floor(simPart->m_animT * m_particleParams.m_uvAnimParams.m_totalAnimFrames);
vFrame0 = Rv(0.0);
while(uFrame0 >= m_particleParams.m_uvAnimParams.m_numAnimFramesU)
{
	uFrame0 -= m_particleParams.m_uvAnimParams.m_numAnimFramesU;
	vFrame0 += Rv(1.0);
}

uFrame1 = uFrame0 + Rv(1.0);
vFrame1 = vFrame0;
if(uFrame1 >= m_particleParams.m_uvAnimParams.m_numAnimFramesU)
{
	uFrame1 -= m_particleParams.m_uvAnimParams.m_numAnimFramesU;
	vFrame1 += Rv(1.0);
}
if(vFrame1 == m_particleParams.m_uvAnimParams.m_numAnimFramesV)
{
	vFrame1 = Rv(0.0);
}
*/



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

struct PSINPUT
{
	float4 texCoord01					: TEXCOORD0;
	float texLerp						: TEXCOORD1;
	float4 colour						: TEXCOORD2;												
};

struct PSINPUT_DEPTH
{
	float4 texCoord01					: TEXCOORD0;
	float texLerp						: TEXCOORD1;
	float3 depthTexCoordPixelDepth		: TEXCOORD2;											
	float4 colour						: TEXCOORD3;	
#ifdef PIXEL_OFFSET_DEPTH_FADE
	float2 vertOfs						: TEXCOORD4;		
#endif //#ifdef PIXEL_OFFSET_DEPTH_FADE
#ifdef PIXEL_ALPHA_DEPTH_FADE
	float particleRadius				: TEXCOORD4;	
#endif //#ifdef PIXEL_ALPHA_DEPTH_FADE
};

struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

// TODO: set this in code as input into shader compilation, if needed
// #define _DEPTH_FROM_ZBUFFER_

// Platform specific conversion of depth buffer into depth value.
// Matches DepthOfField.fx / VisibilityProbe.fx

float	ComputeDepth( float4 depthTex )
{
	const float TWO_0 = 1.0f;
	const float TWO_8 = 256.0f;
	const float TWO_16 = 65536.0f;
	const float TWO_24 = 16777216.0f;

#if defined(_PS3_)

	const float3 FACTORS = float3( ( TWO_16 / ( TWO_24 - 1.0f ) ),
																 ( TWO_8  / ( TWO_24 - 1.0f ) ),
																 ( TWO_0  / ( TWO_24 - 1.0f ) ) );
	float3 rescaled = round( float3( depthTex.a, depthTex.r, depthTex.g ) * 255.0f );
	float depth = dot( rescaled, FACTORS );	
	return depth;

#elif defined(_XBOX)

	float	depth = depthTex.x;
	return depth;

#elif defined(_DEPTH_FROM_ZBUFFER_)

	const float3 FACTORS = float3( ( ( TWO_8 - 1.0f ) / TWO_8 ),
																 ( ( TWO_8 - 1.0f ) / TWO_16 ),
																 ( ( TWO_8 - 1.0f ) / TWO_24 ) );
	float depth = dot( depthTex.xyz, FACTORS );
	return depth;

#else

	float	depth = depthTex.x;
	return depth;

#endif
}


PSOUTPUT fragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;	

	// Read textures
	float4 diffuseTex0Colour = tex2D( diffuseMap0, _input.texCoord01.xy );
	float4 diffuseTex1Colour = tex2D( diffuseMap1, _input.texCoord01.zw );
		
	// modulate with interpolated vertex colour and texture lerp factor (note tex order)
	_output.colour  = ( ( diffuseTex1Colour * _input.texLerp ) + ( ( 1.0f - _input.texLerp ) * diffuseTex0Colour ) ) * _input.colour;
	
	return _output;
}

float Contrast(float Input, float ContrastPower)
{
#if 1
     //piecewise contrast function
     bool IsAboveHalf = Input > 0.5 ;
     float ToRaise = saturate(2*(IsAboveHalf ? 1-Input : Input));
     float Output = 0.5*pow(ToRaise, ContrastPower); 
     Output = IsAboveHalf ? 1-Output : Output;
     return Output;
#else
    // another solution to create a kind of contrast function
    return 1.0 - exp2(-2*pow(2.0*saturate(Input), ContrastPower));
#endif
}

// TODO: another version of the PS that takes into account particle size when calculating the depth fade. Will be needed for situations 
// where we want small particles, e.g. smoke in wheelarches, else they will always be faded when close to car. Method something like this:-
// 1) Estimate thickness of particle by particle diameter (passed from VS) * particle alpha at pixel
// 2) use this instead of MAX_DIFF when calculating depthFade
// 3) Possibly also try clamping 2) to MAX_DIFF? Needed?
//
// Ideas: particle alpha mighe be linear gradient when authored from artist. Use a texture channel to define particle thickness? 
// Surface normal map
//
PSOUTPUT fragmentShaderZ( PSINPUT_DEPTH _input )
{
	PSOUTPUT _output;	

	// Read textures
	float4 diffuseTex0Colour = tex2D( diffuseMap0, _input.texCoord01.xy );
	float4 diffuseTex1Colour = tex2D( diffuseMap1, _input.texCoord01.zw );
	float4 diffuseCombined	 = ( diffuseTex1Colour * _input.texLerp ) + ( ( 1.0f - _input.texLerp ) * diffuseTex0Colour );
	float4 depthTex			 = tex2D( depthMap, _input.depthTexCoordPixelDepth.xy );
		
	float worldDepth = ComputeDepth(depthTex);
	if(worldDepth == 0.0f)
	{
		// special case, sky is depth 0
		worldDepth = 1.0f;
	}
	
#ifdef PARTICLE_PIXEL_DEPTH_IN_CLIP_PLANE_RANGE
	// move from NDC back to camera space, so we can find a linear diff with particle z and use it to scale meaningfully
	worldDepth = nearFarClip.z / (nearFarClip.y - (worldDepth * nearFarClip.w));
#endif //#ifdef PARTICLE_PIXEL_DEPTH_IN_CLIP_PLANE_RANGE
	
#ifdef CONSTANT_DEPTH_FADE
	float diff = worldDepth - _input.depthTexCoordPixelDepth.z;
#define MAX_DIFF	1.0f
	float depthFade = saturate(diff / MAX_DIFF);
#endif //#ifdef CONSTANT_DEPTH_FADE

#ifdef PIXEL_OFFSET_DEPTH_FADE
	float diff = worldDepth - _input.depthTexCoordPixelDepth.z;
	float depthFade = saturate(diff / length(_input.vertOfs));
#endif //#ifdef PIXEL_OFFSET_DEPTH_FADE

#ifdef PIXEL_ALPHA_DEPTH_FADE
	float diff = worldDepth - _input.depthTexCoordPixelDepth.z;
	float depthFade = saturate(diff / (diffuseCombined.a * _input.particleRadius));
	depthFade = Contrast(depthFade, 2.0);
#endif //#ifdef PIXEL_ALPHA_DEPTH_FADE
	
	// modulate with interpolated vertex colour and texture lerp factor (note tex order)
	_output.colour = diffuseCombined * _input.colour;
	_output.colour.a *= depthFade;
	
#ifdef VISUALISE_DEPTH_DIFFERENCE
#ifdef PARTICLE_PIXEL_DEPTH_IN_CLIP_PLANE_RANGE
	_output.colour.r = depthFade;
	_output.colour.g = depthFade;
	_output.colour.b = depthFade;
	_output.colour.a = 1.0f;
#endif //#ifdef PARTICLE_PIXEL_DEPTH_IN_CLIP_PLANE_RANGE
#endif //VISUALISE_DEPTH_DIFFERENCE

#ifdef VISUALISE_PARTICLE_DEPTH
	_output.colour.r = _input.depthTexCoordPixelDepth.z / nearFarClip.y;
	_output.colour.g = _input.depthTexCoordPixelDepth.z / nearFarClip.y;
	_output.colour.b = _input.depthTexCoordPixelDepth.z / nearFarClip.y;
	_output.colour.a = 1.0f;
#endif //#ifdef VISUALISE_PARTICLE_DEPTH
	
#ifdef VISUALISE_ZBUFFER
#ifdef PARTICLE_PIXEL_DEPTH_IN_CLIP_PLANE_RANGE
	_output.colour.r = worldDepth / nearFarClip.y;
	_output.colour.g = worldDepth / nearFarClip.y;
	_output.colour.b = worldDepth / nearFarClip.y;
	_output.colour.a = 1.0f;
#else
	_output.colour.r = worldDepth;
	_output.colour.g = worldDepth;
	_output.colour.b = worldDepth;
	_output.colour.a = 1.0f;
#endif //#ifdef PARTICLE_PIXEL_DEPTH_IN_CLIP_PLANE_RANGE
#endif //#ifdef VISUALISE_ZBUFFER

	return _output;
}



//-----------------------------------------------------------------------
//
// Technique(s)
//

technique BillboardAnimated
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "BillboardAnimated";
	int    normalDeferredID		= 3;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "BillboardAnimated";
	int    lowDetailDeferredID	= 3;
>
{
	pass Pass0
	{
#ifdef _PS3_
		VertexShader = compile sce_vp_rsx VShader();
		PixelShader = compile sce_fp_rsx fragmentShader();
#else
		VertexShader = compile vs_3_0 VShader();
		PixelShader = compile ps_3_0 fragmentShader();
#endif
	}
}

technique BillboardAnimatedCamFade
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "BillboardAnimatedCamFade";
	int    normalDeferredID		= 3;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "BillboardAnimatedCamFade";
	int    lowDetailDeferredID	= 3;
>
{
	pass Pass0
	{
#ifdef _PS3_
		VertexShader = compile sce_vp_rsx VShaderCamFade();
		PixelShader = compile sce_fp_rsx fragmentShader();
#else
		VertexShader = compile vs_3_0 VShaderCamFade();
		PixelShader = compile ps_3_0 fragmentShader();
#endif
	}
}

technique BillboardAnimatedZ
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "BillboardAnimatedZ";
	int    normalDeferredID		= 3;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "BillboardAnimated";
	int    lowDetailDeferredID	= 3;
>
{
	pass Pass0
	{
#ifdef _PS3_
		VertexShader = compile sce_vp_rsx VShaderZ();
		PixelShader = compile sce_fp_rsx fragmentShaderZ();
#else
		VertexShader = compile vs_3_0 VShaderZ();
		PixelShader = compile ps_3_0 fragmentShaderZ();
#endif
	}
}

technique BillboardAnimatedCamFadeZ
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "BillboardAnimatedCamFadeZ";
	int    normalDeferredID		= 3;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "BillboardAnimatedCamFade";
	int    lowDetailDeferredID	= 3;
>
{
	pass Pass0
	{
#ifdef _PS3_
		VertexShader = compile sce_vp_rsx VShaderCamFadeZ();
		PixelShader = compile sce_fp_rsx fragmentShaderZ();
#else
		VertexShader = compile vs_3_0 VShaderCamFadeZ();
		PixelShader = compile ps_3_0 fragmentShaderZ();
#endif
	}
}

